<?php
// Generates images ONLY from the hard-coded reference (REF_IMAGE_PATH).
// Accepts JSON: { prompt: string }
// Returns JSON: { image: base64, format: 'png', file_path: string, file_url?: string }
header('Content-Type: application/json');
header('Cache-Control: no-store');

require_once __DIR__ . '/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['error' => 'Method not allowed']);
  exit;
}

$input  = json_decode(file_get_contents('php://input'), true) ?? [];
$prompt = trim((string)($input['prompt'] ?? ''));
if ($prompt === '') {
  http_response_code(400);
  echo json_encode(['error' => 'Prompt is required']);
  exit;
}

// Load the hard-coded image
list($b64, $err) = load_file_base64(REF_IMAGE_PATH);
if ($err) {
  http_response_code(500);
  echo json_encode(['error' => $err]);
  exit;
}

// Build Venice payload
$payload = [
  'prompt' => $prompt . ' Maintain the exact facial identity and likeness of the reference photo.',
  'image'  => $b64
];

// Call Venice /image/edit (expects raw PNG back)
$res = venice_api_call_binary('/image/edit', $payload);
if (!$res['ok']) {
  http_response_code($res['status'] ?: 500);
  $msg = is_string($res['error']) ? $res['error'] : 'Unknown error';
  echo json_encode(['error' => $msg, 'status' => $res['status']]);
  exit;
}

// --- Save to disk ---
$bytes = $res['data'];                       // raw PNG bytes
$dir   = __DIR__ . '/generated';             // change if you want a different folder
if (!is_dir($dir)) {
  @mkdir($dir, 0775, true);
}
$fname = sprintf('%s_%s.png', date('Ymd_His'), bin2hex(random_bytes(4)));
$path  = $dir . '/' . $fname;

if (file_put_contents($path, $bytes) === false) {
  http_response_code(500);
  echo json_encode(['error' => 'Failed to save generated file on server']);
  exit;
}

// Optional: if your site has a known public URL for this folder, set it here
// e.g. $baseUrl = 'https://example.com/your-app/generated';
$baseUrl = null; // set to your public base URL for the /generated folder (or leave null)
$fileUrl = $baseUrl ? rtrim($baseUrl, '/') . '/' . rawurlencode($fname) : null;

// Also return base64 for instant preview
$out_b64 = base64_encode($bytes);
echo json_encode([
  'image'     => $out_b64,
  'format'    => 'png',
  'file_path' => $path,
  'file_url'  => $fileUrl
]);
?>