<?php
header('Content-Type: application/json');
header('Cache-Control: no-store');
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

// only POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405); echo json_encode(['error' => 'Method not allowed']); exit;
}

// optional style/vibe (form or JSON)
$style = trim((string)($_POST['style'] ?? ($_GET['style'] ?? 'friendly, professional, modern')));
$vibe  = trim((string)($_POST['vibe']  ?? ($_GET['vibe']  ?? 'warm, confident, genuine')));

// 1) Ask Venice chat for profile JSON (structured responses)
$system = "You create concise, wholesome dating profiles. Output only valid JSON that matches the provided JSON schema.";
$user   = "Create one female dating profile. Keep it friendly and SFW.

Return JSON with fields:
- name (string, first name only)
- age (number, 21-50)
- location (string, city + country)
- occupation (string)
- bio (string, 40-80 words, first person)
- desires (string, 25-60 words, what she wants in a relationship)
- seeking (string, 1-2 sentences, who she wants to meet)

Seed preferences: " . json_encode(['style'=>$style,'vibe'=>$vibe]);

$payload = [
  'model' => 'venice-uncensored',
  'messages' => [
    ['role' => 'system', 'content' => $system],
    ['role' => 'user',   'content' => $user]
  ],
  'response_format' => [
    'type' => 'json_schema',
    'json_schema' => [
      'name' => 'dating_profile',
      'strict' => true,
      'schema' => [
        'type' => 'object',
        'properties' => [
          'name' => ['type' => 'string'],
          'age' => ['type' => 'number'],
          'location' => ['type' => 'string'],
          'occupation' => ['type' => 'string'],
          'bio' => ['type' => 'string'],
          'desires' => ['type' => 'string'],
          'seeking' => ['type' => 'string']
        ],
        'required' => ['name','age','location','occupation','bio','desires','seeking'],
        'additionalProperties' => false
      ]
    ]
  ]
];

$res = venice_api_call_json('/chat/completions', $payload, 'POST');
if (!$res['ok']) {
  http_response_code($res['status'] ?: 500);
  $msg = is_array($res['error']) ? json_encode($res['error']) : $res['error'];
  echo json_encode(['error' => $msg, 'status' => $res['status']]); exit;
}

$content = $res['data']['choices'][0]['message']['content'] ?? '';
$profile = json_decode($content, true);
if (!$profile) {
  // attempt to clean
  $clean = preg_replace('/^[^{]+/','', trim($content));
  $clean = preg_replace('/}[^}]*$/','}', $clean);
  $profile = json_decode($clean, true);
}
if (!$profile) {
  http_response_code(500); echo json_encode(['error'=>'Failed to parse profile JSON']); exit;
}

// 2) Generate image with Venice image API and save to /uploads/profiles
$imgPrompt = sprintf(
  "Portrait photography, 3/4 head-and-shoulders of a smiling adult woman, natural lighting, soft background bokeh. Subject vibe: %s. Keywords: %s. No text, no watermark.",
  $vibe ?: 'warm, confident, genuine',
  $profile['occupation'] ?? 'professional'
);
$imgPayload = [
  'model' => 'hidream',
  'prompt' => $imgPrompt,
  'format' => 'png',
  'width' => 1024,
  'height' => 1024,
  'safe_mode' => true,
  'variants' => 1,
  'return_binary' => false
];
$res2 = venice_api_call_json('/image/generate', $imgPayload, 'POST');
if (!$res2['ok']) {
  http_response_code($res2['status'] ?: 500);
  $msg = is_array($res2['error']) ? json_encode($res2['error']) : $res2['error'];
  echo json_encode(['error' => $msg, 'status' => $res2['status']]); exit;
}
$b64 = $res2['data']['images'][0] ?? null;
if (!$b64) { http_response_code(500); echo json_encode(['error'=>'Image not returned']); exit; }
$bin = base64_decode($b64);
if (!$bin) { http_response_code(500); echo json_encode(['error'=>'Image decode failed']); exit; }

$slug = strtolower(preg_replace('/[^a-z0-9]+/i','-', $profile['name'])) . '-' . time();
$filename = $slug . '.png';
$dir = __DIR__ . '/uploads/profiles';
if (!is_dir($dir)) { mkdir($dir, 0775, true); }
file_put_contents($dir . '/' . $filename, $bin);

// 3) Save to DB
try {
  $pdo = pdoConn();
  $stmt = $pdo->prepare("INSERT INTO profiles (name, age, location, occupation, bio, desires, seeking, image_filename) VALUES (:name,:age,:location,:occupation,:bio,:desires,:seeking,:image)");
  $stmt->execute([
    ':name' => (string)$profile['name'],
    ':age' => (int)$profile['age'],
    ':location' => (string)$profile['location'],
    ':occupation' => (string)$profile['occupation'],
    ':bio' => (string)$profile['bio'],
    ':desires' => (string)$profile['desires'],
    ':seeking' => (string)$profile['seeking'],
    ':image' => $filename
  ]);
  $id = (int)$pdo->lastInsertId();
} catch (Throwable $e) {
  http_response_code(500); echo json_encode(['error'=>'DB error: '.$e->getMessage()]); exit;
}

// 4) Respond
echo json_encode([
  'ok' => true,
  'id' => $id,
  'profile' => $profile,
  'image_filename' => $filename,
  'image_url' => 'uploads/profiles/' . $filename
], JSON_PRETTY_PRINT);
