<?php
// send-message-venice-si.php
// Backend for Venice.ai API with strong Sinhala enforcement.
// Endpoint used: https://api.venice.ai/api/v1/chat/completions
// Docs: https://docs.venice.ai/api-reference/endpoint/chat/completions
//
// HOW TO USE:
// - Set VENICE_API_KEY in your server environment, or put it in $API_KEY below.
// - Optional: set VENICE_BASE_URL (defaults to https://api.venice.ai).
// - POST JSON: { message, reset?, model?, lang?, system?, venice_parameters? }
// - Returns: { reply, model }
//
// SECURITY NOTE: Do NOT expose this file publicly without rate limiting / auth if needed.

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();

// ---- Config ----
$BASE_URL = rtrim(getenv('VENICE_BASE_URL') ?: 'https://api.venice.ai', '/'); // keep default
$API_KEY  = getenv('c5HJTclK01-HSaWbjl-L5QdXO85gBmZtFhJnvdEuNQ') ?: 'c5HJTclK01-HSaWbjl-L5QdXO85gBmZtFhJnvdEuNQ';          // <-- put your key or set env
$MODEL_DEFAULT = 'venice-uncensored'; // sensible high-quality default on Venice (can be venice-uncensored)
$TIMEOUT = 45;

// ---- Input ----
$input = json_decode(file_get_contents('php://input'), true) ?: [];
if (isset($input['reset']) && $input['reset']) {
  $_SESSION['si_chat'] = [];
  echo json_encode(['ok' => true, 'reset' => true]); exit;
}

$userMsg = trim((string)($input['message'] ?? ''));
if ($userMsg === '') {
  http_response_code(400);
  echo json_encode(['error' => 'Empty message']); exit;
}

$model = trim((string)($input['model'] ?? $MODEL_DEFAULT));
$clientLang = $input['lang'] ?? 'si-LK';
$userSystem = $input['system'] ?? '';
$veniceParams = isset($input['venice_parameters']) && is_array($input['venice_parameters'])
  ? $input['venice_parameters'] : null;

// ---- Strong Sinhala system prompt + few-shot ----
$SYSTEM_SINHALA = <<<SYS
ඔබ සිංහලෙන් පමණක් සිතන සහ කතා කරන උපකාරකයෙකි.
Unicode සිංහල අකුරු පමණක් භාවිතා කරන්න. Singlish (ලතින්) භාවිතා නොකරන්න.
කෙටියෙන්, පැහැදිලිව, හිතකර ශ්‍රී ලාංකේය රීතියෙන් පිළිතුරු දෙන්න.
පරිශීලකයා ඉංග්‍රීසියෙන් පැවසුවහොත් ද, ඔබ සිංහලෙන් පමණක් පිළිතුරු දෙන්න.
SYS;

$FEWSHOT = [
  ['role'=>'user', 'content'=>'Explain in 1–2 lines: benefits of sleep.'],
  ['role'=>'assistant','content'=>'නිතර හොඳ නින්ද ළඟා වීම ශරීරය ප්‍රතිසාධනය කර ආතතිය, දිරා ගිය ඇල්පතාව අඩු කරයි.\nඉදිරි දවසට අවධානය, මතක ශක්තිය, ආරක්ෂාව වර්ධනය කරයි.']
];

if ($userSystem) $SYSTEM_SINHALA .= "\n\nඅමතර නියමය: ".$userSystem;

// ---- History ----
if (!isset($_SESSION['si_chat'])) $_SESSION['si_chat'] = [];
$history = $_SESSION['si_chat'];

// ---- Build first-pass messages ----
$messages = array_merge(
  [['role'=>'system','content'=>$SYSTEM_SINHALA]],
  $FEWSHOT,
  $history,
  [['role'=>'user','content'=>$userMsg]]
);

// ---- Call Venice (pass 1) ----
$reply1 = venice_chat($BASE_URL, $API_KEY, $model, $messages, $veniceParams, $TIMEOUT);
if ($reply1 === null) {
  http_response_code(502);
  echo json_encode(['error'=>'Venice API call failed']); exit;
}

// ---- Enforce Sinhala: if reply has too little Sinhala, fix via pass 2 ----
if (!looks_sinhala_enough($reply1)) {
  $transSys = "ඔබ වෘත්තීය පරිවර්තකයෙක්. පහත පෙළ ‘Unicode සිංහල’ භාවිතයෙන් ස්වභාවික, පැහැදිලි, කෙටි වාක්‍ය සමඟ ලබා දෙන්න. අමතර විස්තර/ඉංග්‍රීසි නොදමන්න. සිංහල පමණක්.";
  $messages2 = [
    ['role'=>'system','content'=>$transSys],
    ['role'=>'user','content'=>"පහත පෙළ සිංහලට පරිවර්තනය කර ගුණාත්මක ලෙස මෘදුපෙන් රචಿಸಿ:\n\n".$reply1]
  ];
  $reply2 = venice_chat($BASE_URL, $API_KEY, $model, $messages2, $veniceParams, $TIMEOUT);
  if ($reply2) $reply1 = $reply2;
}

// ---- Save history (last ~8 turns) ----
$history[] = ['role'=>'user','content'=>$userMsg];
$history[] = ['role'=>'assistant','content'=>$reply1];
$_SESSION['si_chat'] = array_slice($history, -16);

// ---- Respond ----
echo json_encode(['reply'=>$reply1, 'model'=>$model], JSON_UNESCAPED_UNICODE);
exit;

// ===== Helpers =====

function looks_sinhala_enough($text) {
  // Count Sinhala code points vs letters; require >=45% Sinhala to accept
  $totalLetters = preg_match_all('/\p{L}/u', $text, $m);
  $siLetters    = preg_match_all('/[\x{0D80}-\x{0DFF}]/u', $text, $m2);
  if ($totalLetters <= 0) return false;
  $ratio = $siLetters / max(1,$totalLetters);
  return $ratio >= 0.45;
}

function venice_chat($baseUrl, $apiKey, $model, $messages, $veniceParams = null, $timeout = 45) {
  // Venice chat completions endpoint
  $url = rtrim($baseUrl, '/') . '/api/v1/chat/completions';

  // Venice prefers max_completion_tokens (not max_tokens)
  $payload = [
    'model' => $model,
    'messages' => $messages,
    'temperature' => 0.7,
    'max_completion_tokens' => 512
  ];
  if ($veniceParams) $payload['venice_parameters'] = $veniceParams;

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
      'Content-Type: application/json',
      'Authorization: Bearer '.$apiKey
    ],
    CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE),
    CURLOPT_TIMEOUT => $timeout
  ]);
  $res = curl_exec($ch);
  $err = curl_error($ch);
  $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($res === false || $status < 200 || $status >= 300) {
    // Uncomment for debugging:
    // error_log("Venice error ($status): $err | $res");
    return null;
  }
  $data = json_decode($res, true);

  // Typical Venice (OpenAI-like) schema
  if (isset($data['choices'][0]['message']['content'])) {
    return trim($data['choices'][0]['message']['content']);
  }
  if (isset($data['choices'][0]['text'])) {
    return trim($data['choices'][0]['text']);
  }

  return null;
}
